'use strict';

import React, {Component} from 'react';
import {
  AppRegistry,
  StyleSheet,
  Text,
  View,
  Button,
  TouchableHighlight
} from 'react-native';

import {NotificationsAndroid, PendingNotifications} from 'react-native-notifications';

let mainScreen;

function onPushRegistered() {
  if (mainScreen) {
    mainScreen.onPushRegistered();
  }
}

function onNotificationOpened(notification) {
  if (mainScreen) {
    mainScreen.onNotificationOpened(notification)
  }
}

function onNotificationReceived(notification) {
  if (mainScreen) {
    mainScreen.onNotificationReceived(notification)
  }
}

// It's highly recommended to keep listeners registration at global scope rather than at screen-scope seeing that
// component mount and unmount lifecycle tends to be asymmetric!
NotificationsAndroid.setRegistrationTokenUpdateListener(onPushRegistered);
NotificationsAndroid.setNotificationOpenedListener(onNotificationOpened);
NotificationsAndroid.setNotificationReceivedListener(onNotificationReceived);

const styles = StyleSheet.create({
  container: {
    flex: 1,
    justifyContent: 'center',
  },
  titleText: {
    fontSize: 24,
    textAlign: 'center',
    margin: 10,
  },
  bodyText: {
    fontSize: 18,
    textAlign: 'center',
    margin: 10,
  },
  mainButtonText: {
    fontSize: 25,
    fontStyle: 'italic',
    fontWeight: 'bold',
    textAlign: 'center',
    margin: 10,
  },
  plainButtonText: {
    fontSize: 18,
    fontStyle: 'italic',
    textAlign: 'center',
    margin: 10,
  },
});

class MainComponent extends Component {

  constructor(props) {
    super(props);

    this.onPostNotification = this.onPostNotification.bind(this);
    this.onCancelNotification = this.onCancelNotification.bind(this);

    this.state = {
      elapsed: 0,
      lastNotification: undefined
    };

    console.log('ReactScreen', 'ReactScreen');
    mainScreen = this;

    setInterval(this.onTick.bind(this), 1000);
  }

  componentDidMount() {
    console.log('ReactScreen', 'componentDidMount');
    PendingNotifications.getInitialNotification()
      .then((notification) => {console.log("getInitialNotification:", notification); this.setState({initialNotification: (notification ? notification.getData() : undefined)});})
      .catch((err) => console.error("getInitialNotifiation failed", err));
  }

  componentWillUnmount() {
    console.log('ReactScreen', 'componentWillUnmount');
  }

  onTick() {
    this.setState({elapsed: this.state.elapsed + 1});
  }

  onPostNotification() {
    this.lastNotificationId = NotificationsAndroid.localNotification({title: "Local notification", body: "This notification was generated by the app!"});
  }

  onCancelNotification() {
    if (this.lastNotificationId) {
      NotificationsAndroid.cancelLocalNotification(this.lastNotificationId);
      this.lastNotificationId = undefined;
    }
  }

  render() {
    return (
      <View style={styles.container}>
        <Text style={styles.titleText}>Wix React Native Notifications</Text>
        <Text style={styles.bodyText}>{this.state.initialNotification ? 'Opened from notification' : ''}</Text>
        <Text style={styles.bodyText}>Last notification: {this.state.lastNotification ? '\n'+this.state.lastNotification.body + ` (opened at ''${this.state.notificationRxTime})` : "N/A"}</Text>
        <Text style={styles.bodyText}>Time elapsed: {this.state.elapsed}</Text>
        <Text>{"\n\n"}</Text>
        <TouchableHighlight onPress={() => this.onPostNotification()}>
          <Text style={styles.mainButtonText}>Try Me!</Text>
        </TouchableHighlight>
        <TouchableHighlight onPress={() => this.onCancelNotification()}>
          <Text style={styles.plainButtonText}>Undo last</Text>
        </TouchableHighlight>
        <TouchableHighlight onPress={() => this.onCheckPermissions()}>
          <Text style={styles.plainButtonText}>Check permissions</Text>
        </TouchableHighlight>
        <Button title={'Send local notification'} onPress={this.sendLocalNotification} testID={'sendLocalNotification'}/>
      </View>
    )
  }

  async onCheckPermissions() {
    const hasPermissions = await NotificationsAndroid.isRegisteredForRemoteNotifications();
    if (hasPermissions) {
      alert('Yay! You have permissions');
    } else {
      alert('Boo! You don\'t have permissions');
    }
  }

  sendLocalNotification() {
    NotificationsAndroid.localNotification({
      title: "Local notification",
      body: "This notification was generated by the app!",
      extra: "data"
    });
  }

  onPushRegistered() {
  }

  onNotificationOpened(notification) {
    console.log("onNotificationOpened: ", notification);
    this.setState({lastNotification: notification.getData(), notificationRxTime: this.state.elapsed});
  }

  onNotificationReceived(notification) {
    console.log("onNotificationReceived: ", notification);
  }
}

AppRegistry.registerComponent('WixRNNotifications', () => MainComponent);
