package org.ylhealth.ym.springtest.hibernate;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import javax.persistence.Id;
import org.hibernate.SQLQuery;
import org.hibernate.StatelessSession;
import org.hibernate.engine.spi.SessionFactoryImplementor;
import org.hibernate.event.spi.PostLoadEvent;
import org.hibernate.event.spi.PostLoadEventListener;
import org.hibernate.transform.Transformers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.i18n.LocaleContextHolder;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Interner;
import com.google.common.collect.Interners;

/** */
public class I18NReadEventListener implements PostLoadEventListener {
  /** */
  private static final long serialVersionUID = -7374424944903271183L;

  private Logger logger = LoggerFactory.getLogger(getClass());

  private SessionFactoryImplementor sessionFactory;

  private Interner<String> pool = Interners.newWeakInterner();

  @SuppressWarnings("rawtypes")
  private Cache<String, Map> cache =
      CacheBuilder.newBuilder().expireAfterAccess(1, TimeUnit.MINUTES).build();

  public I18NReadEventListener(SessionFactoryImplementor sessionFactory) {
    this.sessionFactory = sessionFactory;
  }

  @SuppressWarnings("rawtypes")
  @Override
  public void onPostLoad(PostLoadEvent event) {
    Object entity = event.getEntity();
    Class clazz = entity.getClass();

    I18NTranslates i18n = entity.getClass().getAnnotation(I18NTranslates.class);
    if (i18n != null) {
      try (StatelessSession session =
          sessionFactory.openStatelessSession(event.getSession().connection())) {

        Map<String, String> translate = getTranslate(session, i18n, entity, clazz);
        if (translate != null && !translate.isEmpty()) {
          Field[] fields = clazz.getDeclaredFields();
          for (I18NTranslate i18nTranslate : i18n.mapping()) {
            for (Field field : fields) {
              if (field.getName().equals(i18nTranslate.field())) {
                field.setAccessible(true);
                field.set(entity, translate.get(i18nTranslate.column()));
                break;
              }
            }
          }
        }
      } catch (Exception e) {
        logger.error("ReadEventListener error", e);
        throw new IllegalStateException();
      }
    }
  }

  /**
   * @param i18n
   * @param entity
   * @param clazz
   * @return
   * @throws IllegalAccessException
   * @throws InvocationTargetException
   */
  private Map<String, String> getTranslate(
      StatelessSession session, I18NTranslates i18n, Object entity, @SuppressWarnings("rawtypes") Class clazz)
      throws IllegalAccessException, InvocationTargetException {
    Object id = getPKValue(clazz, entity);
    String language = LocaleContextHolder.getLocale().toString();
    String table = i18n.table();
    String idColumn = i18n.id();

    Map<String, Map<String, String>> translateTable =
        translateTable(session, table, language, i18n.mapping(), idColumn);

    return translateTable.get(id);
  }

  @SuppressWarnings("unchecked")
  private Map<String, Map<String, String>> translateTable(
      StatelessSession session,
      String table,
      String language,
      I18NTranslate[] mapping,
      String idColumn) {
    String key = "Table:" + table + "_" + language;
    Map<String, Map<String, String>> result = cache.getIfPresent(key);
    if (result == null) {
      synchronized (pool.intern(key)) {
        result = cache.getIfPresent(key);
        if (result == null) {
          logger.trace("translate query, table: {}", table);
          String sql = String.format("select * from %s where languageCode = ?", table);
          SQLQuery query1 = session.createSQLQuery(sql);
          query1.setResultTransformer(Transformers.ALIAS_TO_ENTITY_MAP);
          query1.addScalar(idColumn);
          query1.addScalar("languageCode");
          for (I18NTranslate i18nTranslate : mapping) {
            query1.addScalar(i18nTranslate.column());
          }
          query1.setParameter(0, language);
          List<Map<String, String>> result1 = query1.list();
          result = new HashMap<>();
          for (Map<String, String> map : result1) {
            map.remove("languageCode");
            result.put(map.get(idColumn), map);
          }
          cache.put(key, result);
        }
      }
    }
    return result;
  }


  private Object getPKValue(Class<?> clazz, Object entity)
      throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
    Collection<Field> fields = getAllDeclaredFields(clazz);
    for (Field field : fields) {
      if (field.getAnnotation(Id.class) != null) {
        field.setAccessible(true);
        return field.get(entity);
      }
    }
    for (Method method : clazz.getDeclaredMethods()) {
      if (method.getAnnotation(Id.class) != null) {
        return method.invoke(entity);
      }
    }
    return null;
  }

  private static Collection<Field> getAllDeclaredFields(Class<?> clazz) {
    List<Field> fields = new ArrayList<>();
    Collections.addAll(fields, clazz.getDeclaredFields());
    if (clazz.getSuperclass() != null) {
      fields.addAll(getAllDeclaredFields(clazz.getSuperclass()));
    }
    return fields;
  }
}
