# Saturate an Image

Effects on images like *saturation, contrast, brightness, inverse, hue,...* are very easy to implement in GLSL. Here is the example of **Saturation**.

```html
<Saturation
  width={256}
  height={171}
  factor={saturationFactor}
  image={{ uri: "http://i.imgur.com/iPKTONG.jpg" }}
/>
```

![](2.gif)


## Implementation

```js
const React = require("react-native");
const GL = require("gl-react-native");

const shaders = GL.Shaders.create({
  saturation: {
    frag: `
precision highp float;
varying vec2 uv;
uniform sampler2D image;
uniform float factor;

void main () {
  vec4 c = texture2D(image, uv);
  // Algorithm from Chapter 16 of OpenGL Shading Language
  const vec3 W = vec3(0.2125, 0.7154, 0.0721);
  gl_FragColor = vec4(mix(vec3(dot(c.rgb, W)), c.rgb, factor), c.a);
}
    `
  }
});

class Saturation extends React.Component {
  render () {
    const { width, height, factor, image } = this.props;
    return <GL.View
      shader={shaders.saturation}
      width={width}
      height={height}
      uniforms={{ factor, image }}
    />;
  }
}
```

GL.View supports `uniforms` props that allow to send to the shader any parameters from the JavaScript world.

In this example, we send the `factor` number (that come from props) and we also send the image.
The image have the same format as the `source` passed in an `<Image/>`.
